;
;  C Startup.asm - Control Cross-C all-purpose startup code.
;
;  Copyright (c) 2002 Softools, Inc.  All rights reserved.
;
;  Since this file is self-configuring using several assemble-time
;  prompts, it should be assembled separately for each application
;  program being developed.  Labels can be defined to bypass the
;  prompts - the WinIDE New Project Wizard in fact does this.
;
;  If a pre-startup module is used to initialize other hardware,
;  interrupt vectors, etc., the global entry point CSTART should
;  be jumped to from that module.  Otherwise, this module is placed
;  at location 0 and will handle the startup.  The state of interrupts
;  is NOT changed by this module.  Therefore, if a pre-startup module
;  turns them off, main () or this module should turn them back on.
;  The SP and a MMU and required Rabbit peripherals are initialized
;  by this module.
;
;  The following memory configuration is used in the Control Cross-C
;  environment.  Note that not all segments are necessarily used in
;  all applications.
;
;  Segment	Contains
;  ==================================================================
;  CODE 	Startup code, libraries and near compiled code.
;  STARTUP	Startup vector table used to dynamically run startup
;		  initialization code.
;  CONST	String literals, const objects and
;		  initializers for automatic aggregates.
;  DATA 	Initialized data area.	May be copied from ROM to RAM.
;  FARCODE	Code compiled as far functions in C.
;  FARDATA	Initialized far data area.  This is stored out of the
;		  logical address space.
;  BSS		Uninitialized data area.  This may be zeroed on startup.
;  FARBSS	Uninitialized far data area.  The startup code DOES NOT
;		  zero this space!
;  BSSZ 	Initialized by 0 data.	This may be zeroed on startup.
;  STACK	The amount reserved for the CPU stack
;		The remmaining area is left over and used for the dynamic heap.
;
;  --------------------------------------------------------------------------
;  A program is linked with SLINK using the following commands:
;  This applies to the command line tools only - the WinIDE has
;  a dialog that configures these settings.  The New Project Wizard
;  will create projects and these can be used as a starting point.
;    (This is only an example.	The addresses used need to be changed
;    to reflect the target system and application program.)
;
;  In the Softools WinIDE, the names in parens (e.g. (CODE)) are segment
;  definitions and take the same addresses (e.g. start=8000 sets logical
;  address to 8000).  Modules are included around segment definitions.
;
;  (CODE, start=0)		; Place CODE at 0
;  (CONST)			; Place constants next
;  (STARTUP)			; Special startup vector table
;  (DATA)			; Place DATA next in ROM
;  (FARCODE,overlap)		; Place far C functions next
; *(FARDATA)			; Initializers (should not be in logical space)
;  (IDATA<DATA,start=8000)	; Reserve space for copied initialized DATA
;  (BSS)			; BSS is after DATA (at 8000)
;  (BSSZ)			; BSSZ is after BSS
;  (STACK)			; The CPU stack follows data
; *(FARBSS,start=0,at=x,overlap); FARBSS can be anywhere in physical space
;  program modules and libraries should be listed here
;
; * Used only if 'far' is used in C.
;
;  --------------------------------------------------------------------------
;  The following assembly-time questions are asked to configure the
;    application:
;
;  We prompt for the CPU stack size.  This eliminates editing
;    this file just to change the stack size.
;
;  We prompt to find out if the system is battery backed.
;    If it is, then we do not zero-fill BSS and
;    We prompt if the system needs a power-up test for initialization.
;      If it does, then a variable stored in segment DATA is compared
;      to a known default value.  If the variable contains this value,
;      the initialization is skipped.  Otherwise, it is performed.
;    If not, segment BSS is never zero filled.
;
;  In order to eliminate the prompts below, which may be necessary
;    in the Windows WinIDEbug environment because a build stops for
;    each prompt, the startup questions can be supplid as EQUs.
;
;  If STACK_SIZE is supplied, BATT_BACKED, and BATT_TEST
;    can be supplied to override the questions.  If any but STACK_SIZE
;    is omitted, all others default to N.
;
Y	equ	1
N	equ	0
;
	.ifdef	STACK_SIZE
;
size	.equ	STACK_SIZE	; Application stack size?

	.ifdef	BATT_BACKED
bb_sys	.equ	Y
	.printx Building RAM battery backed C startup
	.else
bb_sys	.equ	N		; Battery backed system?
	.printx Building RAM initializing C startup
	.endif

	.ifdef	BATT_TEST
bb_test .equ	Y
	.printx Building validating RAM battery backed C startup
	.else
bb_test .equ	N	; Do battery backup test? (Don't care if bb_sys is N)
	.endif

	.ifndef	INC_DEBUG
INC_DEBUG .equ	Y
	.endif
	.if	! INC_DEBUG
	.printx Building NO DEBUG region C startup (cannot debug in WinIDEbug)
	.endif

	.ifndef	GEN_CRC
GEN_CRC	.equ	N
	.endif

	.else
;
size::	.input	"Application CPU stack size? "
bb_sys: .input	"Is application running in a battery-backed system (Y/N)? "
	.if	bb_sys
bb_test:.input	"Do data initialization if a battery-backed test fails (Y/N)? "
	.else
bb_test equ	N
	.endif
INC_DEBUG:.input "Include 2k debug kernel region (Y/N)? "
GEN_CRC:.input	"Add CRC table and runtime CRC check program (Y/N)? "
	.endif
;
	.include Rabbit.inc
;
__stack_size::	.equ	even size
;
	.cseg				; Define implicit segment ordering
	.defseg CONST
	.defseg STARTUP
	.dseg
	.defseg	FARCODE
	.defseg	FARCODE2
	.defseg FARCONST
	.bseg
	.defseg BSSZ
	.defseg IDATA,data
	.defseg STACK,stack,align=2	; Define STACK segment
	.defseg FARBSS,data
	.seg	STACK
_stackMarker::
	.ds	4			; A 4-byte marker that can check for stack overruns
	.ds	__stack_size - 4
__stack::.equ	$			; Initial SP
;
	.cseg
	.phase	0
	ljp	cstart			; This is at 0
inFlash:.equ	$-1			; Flash marker byte 3 (must not be 0!)
;
	.org	0Ah
_19200TC:.ds	1			; Bootstrap leaves 19200 TC here
_GCDR:	.ds	1			; And GCDR here.
_TAT4R:	.ds	1			; And TAT4R here.
	.org	0Eh
__CRC_location::
	.ds	2			; Store CRC here at E
;
;  Debug output and input vectors.  The __stdio FILE pointer
;  is set to these entry points.  In production code, printf
;  calls these per character doing nothing.  In debug mode,
;  the debug kernel redirects these to output to the WinIDE.
;
	.org	10h			; Debug in vector
	ret
;
	.org	18h			; Debug out vector
	ret
;
;  Leave a RET at 50h in case a RST 28 is executed.  If setInterruptHandler
;  is called, this vector is copied when IIR is changed still preserving
;  the RET.  The application may still use setInterrupthandler to provide
;  a RST 28 handler so it can handle RST 28s.
;
	.org	50h
	ret
;
;  If INC_DEBUG is defined, cstart is built without a 2k hole normally
;  required for the debug kernel.  If you disable this, you cannot run
;  this in the debugger (even if the debug kernel is loaded).
;
	.if	INC_DEBUG
;
	.org	800h			; 800h is the base above the debug kernel
;
	.else
;
	.org	60h+40			; Leave hole for system ID block
;
	.endif
	.dephase
;
;  idBlock definition, defined as an address
;
_idBlock::	.equ	60h
__idBlockMac1::	.equ	60h+0Bh
__idBlockMac2::	.equ	60h+0Bh+6
;
;  This table defines the area of a program to CRC.  This is done
;  by SLINK automatically upon defining the table and location
;  to store the final CRC.  The code to CRC the program based on
;  this table is included in cstart and included and run if the
;  label GEN_CRC is non-0.  This makes no assumptions about
;  the ordering and size of segments.  The CRC is stored by
;  SLINK (in all data formats written) at __CRC_location.
;
	.if	GEN_CRC
__CRC_table::				; Must have this name to signal SLINK
	.dl	SGAT CODE+810h		; CRC segment CODE (minus first 16 bytes)
	.dl	SGSZ CODE-810h
	.dl	SGAT CONST		; CRC CONST
	.dl	SGSZ CONST
	.dl	SGAT DATA		; CRC DATA
	.dl	SGSZ DATA
	.dl	SGAT FARCODE		; CRC FARCODE
	.dl	SGSZ FARCODE
	.dl	SGAT FARCONST		; CRC FARCONST
	.dl	SGSZ FARCONST
	.dl	0			; Marks end of table
	.endif				; GEN_CRC
;
	.seg	FARCODE
cstart::ljp	..0
..0:	ipset	3			; Being safe
	.debug	{ cstart -1 239 cstart end
	xor	a,a
	ld	iir,a
	ld	sp, __stack
;
;  Initialize the MMU SEGSIZE and STACKSEG to map logical data areas to their
;  physical addresses.
;
        ld	a,(sgst BSS >> 8) & 0F0h
	ioi ld	(SEGSIZE),a
	ld	a,(sgat BSS >> 12) - (sgst BSS >> 12)
	ioi ld	(STACKSEG),a
;
	.if	INC_DEBUG
;
;  Test inFlash byte - will be 0 if the code is in RAM
;
	ld	a,(inFlash)		; Test for flash, skip init if not
	or	a			; Is non-0 in flash
	jp	z,..inRam
;
	.endif
;
;  Initialize Rabbit peripherals here.  This uses a table to
;  quickly load address and data to initialize the Rabbit processor.
;  Simply extend the table as needed.  The Watchdog is set to 2
;  seconds - plenty of time to get to main from here.
;
	ld	bc,..iTblSize		; Size
	ld	hl,..iTbl		; Table
..lp:   ld	e,(hl)			; Addr
	inc	hl
	ioi ldi
	jp	v,..lp
;
	.if	GEN_CRC
;
	lcall	$gencrc##
	jp	z,..cstart
;
;  CRC has failed.  Handling this is program dependent.
;
..CRCfailed:
	nop				; Handle a bad program CRC
	jr	..CRCfailed
;
	.else
;
	jr      ..cstart
;
	.endif		; GEN_CRC
;
;  This defaults to having the Flash at 00000 and RAM at 80000.
;  Note: These do not update the shadow RAM locations.  If you need
;  to readback any registers used here, either initialize the RAM
;  (e.g. _srMMIDR is the MMIDR shadow location) or use the
;  shadow I/O functions in C to reinitialize them.
;  Note: MBxCR register shadows are initialized.
;
;  Defines for MBxCR registers.  Used here and to initialize the shadows.
;  MB0CR must be 0 wait states or the baud rate constants won't be
;  correct. In flash, this starts up in 0 wait states and later if the speed
;  would be too high doubling, it is changed to 1 wait state.  If 2
;  wait states are needed see set1ws below.
;
MB0CRval	.equ	0C8h	; MB0CR: 0 ws, OE0, CS0 active, WP enabled
MB1CRval	.equ	0C8h	; MB1CR: 0 ws, OE0, CS0 active, WP enabled
MB2CRval	.equ	0C5h	; MB2CR: 0 ws, OE1, WE1, CS1 active
MB3CRval	.equ	0C5h	; MB3CR: 0 ws, OE1, WE1, CS1 active
;
;  Note: Be careful using wait states on early Rabbit 2000 processors.
;
..iTbl:
;
;  Set STACKSEG to 80000 - logical start of BSS
;  If RAM begins at C0000 change the 80 to C0.
;
	.db	STACKSEG,80h - (sgst BSS >> 12)
	.db	GCSR,	8	; GCSR: no periodic int, main osc no div
	.db	GCDR,	0
	.db	MMIDR,	0	; MMIDR: normal operation
	.db	MB0CR,	MB0CRval
	.db	MB1CR,	MB1CRval
	.db	MB2CR,  MB2CRval
	.db	MB3CR,	MB3CRval
	.db	TACSR,  001h	; TACSR: Enable timer A
	.db	SPCR,	080h	; SPCR: Disable SMODE pins
	.db     WDTCR,	5Ah	; WDTCR: 2 Seconds to get to main
;
..iTblSize=($-..iTbl)/2
;
;  This table of IO addresses will be zeroed.  This is to ensure
;  that uninitialized Rabbit peripherals and registers come up
;  consistently in Flash and RAM execution.
;
..zTbl:	.db	PADR, PBDR, PCDR, PDDDR, PDDR, PDCR, PDFR, PDDCR
	.db	PEDDR, PEDR, PECR, PEFR, TBCSR, TBCR, IB0CR, IB1CR
	.db	IB2CR, IB3CR, IB4CR, IB5CR, IB6CR, IB7CR, I0CR, I1CR
	.db	TAT1R, SBCR, SCCR, SDCR, GPSCR, BDCR, MECR, PFDR
	.db	PFCR, PFFR, PFDCR, PFDDR, PGDR, PGCR, PGFR, PGDCR
	.db	PGDDR, PWL0R, PWM0R, PWL1R, PWM1R, PWL2R, PWM2R, PWL3R
	.db	PWM3R, ICCSR, ICCR, ICT1R, ICT2R, ICS1R, ICS2R, QDCSR
	.db	QDCR, 0x68, 0x69, 0x6A, 0x6B, 0x6C, 0x6D, 0x6E
	.db	0x6F, 0x78, 0x79, 0x7A, 0x7B, 0x7C, 0x7D, 0x7E
	.db	0x7F, 0xA3, 0xA4, 0xA5, 0xA6, 0xA7, 0xA8, 0xAA
	.db	0xAB, 0xAD, 0xAF, 0xB2, 0xB3, 0xB4, 0xB5, 0xCC
	.db	0xDC, 0xC5, 0xD5, 0xE5, 0xF5, 0xCD, 0xDD
;
..zTblSize=$-..zTbl
;
	.if	INC_DEBUG
;
;  Running in RAM at this point.  The folowing code can
;  be executed but only in RAM debugging - be careful
;  what is placed here - it won't run in the Flashed version.
;
..inRam:ipset	2			; So that serial break-outs work
	xor	a,a			; A NOP
	ld	(10h),a			; In RAM, remove RET to allow printfs
	ld	(18h),a			; Remove this to stub printf in RAM
	.endif
;
;  Zero Rabbit registers in zTbl
;
..cstart:
	xor	a,a
	ld	b,..zTblSize
	ld	hl,..zTbl
..zlp:	ld	e,(hl)
	ioi ld	(de),a
	inc	hl
	djnz	..zlp
;
;  If checking a battery backed system, this code defines in segment
;  DATA a test pattern and checks to see if it valid.  If it is not,
;  then BSS, BSSZ, and DATA are initialized.  If it matches, this
;  bypasses all initialization and continues below.
;
	.if	bb_test
	.dseg
bb_patt:db	0C3h,0F0h,0AAh,03Ch,0F0h,055h
	.seg	FARCODE
	ld	hl,bb_patt		; Test pattern
	ld	b,6			; 6-byte pattern
	ld	de,0
..test: ld	a,d
	add	(hl)
	ld	d,a
	ld	a,e
	xor	(hl)
	ld	e,a
	inc	hl
	djnz	..test			; All bytes XORed is 0
	ld	a,low (0C3h+0F0h+0AAh+03Ch+0F0h+055h)
	sub	d			; Check sum of data
	jr	nz,..fail
	or	e			; Data XORed is 0
	jr	z,..strt		; Skip initialization
..fail: .endif
;
;  Zero out the BSS data area.
;  This is NOT done if the system is battery-backed!
;
	.if	! bb_sys || bb_test
	ld	hl, sgst BSS		; Logical start of BSS
	ld	bc, sgsz BSS		; Size of BSS
..zero:	ld	a,b
	or	c
	jr	z,..bssz
	ld	(hl),0			; 0 first byte
	inc	hl
	dec	bc
	jr	..zero
	.endif
;
;  Zero out the BSSZ data area.  This is for data with initializers of
;  0 (e.g. int x = 0;) and MUST be done even in a battery backed system.
;  (Unless the battery-backed test passes and all initialization is
;  skipped.
;
..bssz: ld	hl, sgst BSSZ		; Start of BSSZ
	ld	bc, sgsz BSSZ		; Size of BSSZ
..loop: ld	a,b
	or	c
	jr	z,..data
	ld	(hl),0			; Zero BSSZ
	inc	hl
	dec	bc
	jr	..loop
;
;  Copy initialized data to segment IDATA.
;
..data:	ld	iy, sgat DATA 		; Phys start of DATA
	ld	e, sgat DATA >> 16
	ld	ix, sgst IDATA 		; Logical start of IDATA
	ld	bc, sgsz DATA		; Size of DATA
..copy:	ld	a,b
	or	c
	jr	z,..strt
	dec	bc
	ld	a,e			; No data
	ldp	hl, (iy)
	ld	(ix), l			; Copy
	inc	ix
	inc	iy
	ld	hl,iy
	bool	hl
	jr	nz,..copy
	inc	e
	jr	..copy
;
;  Unconditionally zero all shadow registers regardless of battery back SRAM.
;
..strt:	ld	b,__endSR##-__startSR##	; Start and end markers
	ld	hl,__startSR##
..0sr:	ld	(hl),0
	inc	hl
	djnz	..0sr
;
;  Read GCSR and store reset code
;
	ioi ld	a,(GCSR)
	rlca
	rlca
	and	a,3
	ld	(_resetCode),a
	ld 	a,1
	ld	(_srTACSR##),a		; Set TACSR (timer A is running)
	ld	(_srTAPR##),a		; Default value on a reset
	ld	a,0C0h
	ld	(_srPBDDR##),a		; Ditto
	add	a,a			; 80h
	ld	(_srSPCR##),a		; Ditto
	ld	a,(inFlash)		; Test for flash, skip init if not
	or	a			; Is non-0 in flash
	jr	nz,..flash
;
;  Initialize GCDR shadow as debug kernel left it and set other shadows
;
	ld	a,(_GCDR)		; Init srGCDR shadow
	ld	(__GCDR),a		; Save for going in and out of doubling
	ioi ld	(GCDR),a		; Restore doubler as on startup
	ld	(_srGCDR##),a
	ld	a,(_TAT4R)
	ld	(_srTAT4R##),a
	ioi ld	(TAT4R),a		; And TAT4R
	ld	a,0C0h			; Init srPCSR
	ld	(_srPCFR##),a
	ld	hl,_srMB0CR##		; Set MBxCR shadows
	ld	(hl),0C5h
	inc	hl
	ld	(hl),0C5h
	inc	hl
	ld	(hl),080h
	inc	hl
	ld	(hl),080h
	ld	a,(_19200TC)
	ld	(__serial19200TC),a	; Save 19200TC
	jr	..skDbl
;
	.ifndef	DISABLE_CLOCK_DOUBLER
;
;  Calculate 19200 TC using that to enable clock doubled mode if the
;  speed allows us to do so.  For low power operation, this can be
;  removed or reversed in main by writing a 0 to GCDR (using a shadow
;  out).  This sets up for a 20nS low time for < 14MHz xtals.
;  Modify the value if this doesn't suit your system.
;
..flash:ld	hl,_srMB0CR##
	ld	(hl),MB0CRval		; Set MBxCR shadows
	inc	hl
	ld	(hl),MB1CRval
	inc	hl
	ld	(hl),MB2CRval
	inc	hl
	ld	(hl),MB3CRval

	call	SerialCalc19200TC
	ld	de,__serial19200TC
	ld	(de),a     		; Save TC
	ld	hl,1			; Enable
	lcall	$enableClockDoubler##
	jp	..done
;
;  Here it's in RAM, if the
;  19200 TC indicates we need wait states, one is added, even
;  in RAM to keep the timing between RAM and flash the same.
;
..skDbl:cp	a,52			; 0 wait state cutoff (30MHz)
	jp	c,..done
	ld	e,MB0CR
	ld	hl,_srMB0CR##
	res	6,(hl)			; Use 7 for 2 wait states
	ioi	ldi
	nop
	res	6,(hl)
	ioi	ldi
;
	.else	; No doubling
;
..skDbl:ld	a,(_GCDR)		; Doubled?
	or	a,a
	jr	z,..done
	xor	a,a
	ioi ld	(GCDR),a
	ld	(_srGCDR##),a
	ld	hl,__serial19200TC
	sra	(hl)			; Halve 19200TC since speed is half
	ld	hl,_srTAT4R##
	inc	(hl)
	sra	(hl)			; Halve divisor since speed is half
	dec	(hl)
	ld	e,TAT4R
	ioi ldi
	ld	e,MB0CR
	ld	hl,_srMB0CR##
	set	6,(hl)			; Use 7 for 2 wait states
	ioi	ldi
	nop
	set	6,(hl)
	ioi	ldi
..flash:
;
	.endif	; DISABLE_CLOCK_DOUBLING
;
..done:
;
;  Get function pointers in segment STARTUP and call each one.
;  This allows compile-time installment of startup initialization
;  which can be executed before main.  The order of call is the
;  order of the addresses in the output.  In a single module they
;  are guaranteed to occur in order.  Modules linked ahead of others
;  are guaranteed to execute their STARTUP functions first.
;  Included only if STARTUP_SEG is defined because not many users use
;  this and therefore the code is wasted.
;
	.ifdef	STARTUP_SEG
	ld	a,sgsz STARTUP / 2	; Number of entries
	or	a
	jr	z,..main
	ld	hl,sgst STARTUP
..glp:	ld	e,(hl)
	inc	hl
	ld	d,(hl)
	inc	hl
	push	hl
	push	af
	call	.callde
	pop	af
	pop	hl
	dec	a
	jr	nz,..glp
	.endif
;
;  Initialize xmem starting address and size.  This can be changed
;  if you want a fixed pool of a fixed size.  It's implemented here
;  with a simple memory sizing routine to maximize xmem in RAM and Flash.
;  This must do some work using the values in the MMU to find the real
;  address of memory since it varies in a RAM and flash enviroment.
;
..main: ld	ix,__xmemStart
	ioi ld	a,(STACKSEG)		; Get physical start of RAM
					; Add in logical adder
	add	a,(sgst stack + sgsz stack) >> 12
	bool	hl
	ld	l,a
	rr	hl
	rr	hl
	rr	hl
	rr	hl
	ld	h,0
	ld	(ix+2),hl		; Set MSW
	ld	(ix+10),hl		; Set heap pointer too
	ld	hl,(sgat stack + sgsz stack) << 4 & 0xFFF0
	rrca
	rr	hl
	rrca
	rr	hl
	rrca
	rr	hl
	rrca
	rr	hl
	ld	iy,hl
	ld	(ix),hl
	ld	(ix+8),hl
;
; Find the base of memory - in AIY
;
	ld	iy,0
	xor	a,a
..flp:	ldp	hl,(iy)
	ex	de,hl
	ld	hl,55AAh
	ldp	(iy),hl
	ldp	hl,(iy)
	ex	de,hl
	ldp	(iy),hl
	ld	hl,55AAh
	or	a,a
	sbc	hl,de
	jr	z,..ram
	ld	de,4000h
	add	iy,de           	; Add 16k
	adc	a,0
	jr	..flp
;
;  Find size of RAM - AIY is the start of RAM.
;
..safe:	.ds	2			; A safe place in case we hit this code
..ram:  ld	de,low ..safe		; Ensure AIY won't hit this code
	add	iy,de
	ldp	hl,(iy)			; Get word
	ex	de',hl			; Save it
	ld	hl,055AAh		; Store 55AA there
	ldp	(iy),hl
	ld	b,0			; Check for up to 256 4k pages
	ld	c,a			; C is base
	altd ld	hl,iy			; Save base
	jr	..sk4k			; Skip 4k or wrap test fails
..xlp:  ldp	hl,(iy)			; Save word
	ex	de,hl
	ld	hl,bc
	ldp	(iy),hl			; Store new word
	ex	af,af'			; Check for wrap
	ld	a,c
	exx
	ldp	hl',(hl)		; Check first page for wrap
	exx
	ld	a,0AAh
	cp	a,l
	jr	nz,..end
	ld	a,055h			; If we wrap we hit his
	cp	a,h
	jr	nz,..end		; Wrapped, then done (restores this word)
	ex	af,af'
	ldp	hl,(iy)
	ex	de,hl
	ldp	(iy),hl			; Restore word
	ex	de,hl
	or	a,a
	sbc	hl,bc			; Same?
	jr	nz,..end		; Jump if not
..sk4k:	ld	de,1000h
	add	iy,de           	; Add 4k
	adc	a,0
	cp	a,10h			; Reached end of space?
	jr	z,..end			; Done if so
	djnz	..xlp			; Try next 16k page
..end:	ex	de',hl			; Fix
	ld	a,c
	ldp	(iy),hl			; Restore first word
	ld	a,(inFlash)
	or	a,a
	ld	a,0
	jr	z,..cram
	sub	a,b			; Number of 4k pages of RAM
	push	af
	sub	a,((sgat stack + sgsz stack) - sgat bss + 0xFFF) / 1000h
	jr	..ok
..cram:	sub	a,b			; Number of 4k pages of RAM
	push	af                      ; Remove code used for RAM load
	sub	a,(sgat stack + sgsz stack + 0xFFF) / 1000h
..ok:	ld	c,a
	ld	b,0
	ld	de,1000h
	mul				; Bytes
	ld	(ix+6),hl		; High word
	ld	hl,bc
	ld	(ix+4),hl
	pop	af
	ld	b,0
	ld	c,a
	mul				; Amount of RAM
	ld	(ix+14),hl
	ld	hl,bc
	ld	(ix+12),hl
;
;  Initialize __stdio to the output vectors.
;  Set the __stack_marker to known values to detect stack overruns.
;  Call main ().  Note, main must be far.  If it is declared near,
;  change the lcall to call and $main to _main.
;
	ld	hl,__debugStdio
	ld	(__stdio),hl
	ld	ix,_stackMarker
	ld	hl,5555h
	ld	(ix),hl
	add	hl,hl			; AAAA
	ld	(ix+2),hl
	lcall	$main## 		; Just halt if it returns
	ld	a,(inFlash)
	or	a,a			; If running in RAM do RST
	jr	nz,$                    ; If running in Flash - hang
_mainExit::
	rst	28h			; break into debugger
;
;  Here, the following routines are stubbed, since there is little that
;  can be done in an embedded system.  If you DO want to handle these
;  somehow, remove the public entry points here.
;
_abort::
	rst	28h			; break into debugger
;
.callde:
	push	de			; Calls address in DE
	ret
;
;  Calculate the timer constant for 19200.
;  Returned in A.
;  This must be called with 0 wait states!
;
SerialCalc19200TC:
	ld	de,1FFFh	; mask for RTC bits
..wait: ioi ld	(RTC0R),a	; load RTC registers
	ioi ld	hl,(RTC0R)	; get LSW RTC
	and	hl,de		; mask off bits
	jr	nz,..wait	; wait bits 0-9 = 0
	ld	a,5Ah
	ld	de,0000h	; DE = count
..loop: inc	de
	ld	b,49
	djnz	$
	ioi ld	(WDTCR),a
	ioi ld	(RTC0R),a
	ioi ld	hl,(RTC0R)	; get LSW RTC
	bit	4,h		; test bit 10
	jr    	z,..loop
	ld	a,d		; 19200 time constant
	ret
end:	.debug	} end 834
;
;  Define variables defining free RAM available for the heap.
;  If the heap is moved, __heap_end must remain >= __heap_start.
;  If malloc/free are not used, then it doesn't matter.  These
;  must address memory that is always mapped in.  It's probable
;  but very important that the heap start is NOT 0.
;
;  We highly recommend against using malloc and free in an
;  embedded application.
;
;  Te define a head, set HEAP_SIZE in the Asm defines for cstart
;  or change the HEAP_SIZE definition below only when HEAP_SIZE
;  isn't already defined.
;
	.ifndef	HEAP_SIZE
HEAP_SIZE = 0
	.endif
;
__heap_start::
        dw      _heap
__heap_end::
        dw      _heap_end
;
        .bseg
_heap:	.ds	HEAP_SIZE
_heap_end:
;
	.bseg
;
;  Calculated time constant for TAT4R.  Use by serial libraries
;  to handle baud rates without regard to CPU speed.  If GCDR
;  is changed, this value must also be updated (going into
;  double clock mode means you must double this constant!)
;  This must be adjusted accordingly if GCDR is changed!
;
__serial19200TC::
	.ds	1
;
;  Reason for reset is stored here:
;    00 - Program restarted at 0
;    01 - Watchdog caused reset
;    03 - Reset line caused reset
;
_resetCode::
	.ds	1
;
;  Byte written to GCDR.  Can be used to go in and out of double speed.
;
__GCDR::
	.ds	1
;
;  Define _stdio here so we don't bring in fassign just to use this variable
;
__stdio::
	.ds	2
;
;  Define xmemory base, size, and heap pointer.  The xmem heap is sized at runtime
;  to allow the maximum xmem space in RAM and flash.  These must be kept together
;  and as defined.
;
__xmemStart::
	.ds	4
__xmemSize::
	.ds	4
__xmemHeap::
	.ds	4
__detectedRAM::
	.ds	4
;
;  An initialized FILE structure for initial printf redirection
;
	.dseg
__debugStdio:
	.dw	18h			; Output address
	.dw	10h			; Input address
	.dw	0			; Putback char
;
	.if	INC_DEBUG
start	=	cstart
	.else
start	=	0xFFFF			; FFFF means no start (stops autorun)
	.endif
	.end	start
